#region References

using System;
using System.IO;
using System.Text;
using System.Data;
using System.Collections;
using gov.va.med.vbecs.Common.Log;

#endregion

namespace gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages
{
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>David Askew</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>5/23/2008</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary>The HL7BtsMessage type represents the HL7 v2.5 Blood Product Transfusion/Disposition message. It implements the HL7ProtocolMessage type and is not meant to be inherited.</summary>

	#endregion 

	/// <summary>
	/// New class for BCE (CR 2962)
	/// </summary>
	public sealed class HL7BtsMessage : HL7ProtocolMessage
	{
		#region Variables

        // Events Logger
        private readonly ILogger _eventsLogger =
            LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

		/// <summary>
		/// Accepted Message Type
		/// </summary>
		private const string BTS = "BTS";

		/// <summary>
		/// Accepted Trigger Event
		/// </summary>
		private const string O31 = "O31";

		private const bool RESPONSE_REQUIRED_INDICATOR = true;

		private char[] _delimiters;

		private string[] _msh;
		private string[] _pid;
		private string[] _pv1;
		private string[] _orc;
		private string[] _bpo;
		private string[] _btx;

		private string _triggerEvent;

		private ArrayList _nte;

		/// <summary>
		/// Use to indicate any error(s) encountered while loading the message.  These codes  
		/// will be returned in a negative acknowledgement to the sending application.
		/// </summary>
		AckErrorCodes _ackErrorCode;

		#endregion

		#region Constructors

		/// <summary>
		/// Empty static constructor added to get rid of "beforefieldinit" attribute generated by compiler.
		/// </summary>
		static HL7BtsMessage() {}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8638"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>HL7BtsMessage</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8639"> 
		///		<ExpectedInput>null</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor to load HL7 message.
		/// </summary>
		/// <param name="message">String representation of the HL7 message.</param>
		public HL7BtsMessage( string message )
			: base( message )
		{
			if( message == null )
			{
				throw( new ArgumentNullException( "message" ) );
			}
			//
			_ackErrorCode = AckErrorCodes.MessageAccepted;
			//
			_message = message;
			//
			try
			{
				// This includes field separator in position [0] and encoding characters in [1] - [4]
				_delimiters = HL7Utility.ParseGetMessageDelimiters(message);			
				//
				// Split the message into segments
				LoadDataFromHL7Message();
				//
				_messageControlID = HL7Utility.GetMessageControlID( message );
				//
				string [] messageTypeAndTriggerEvent = MSH[9].Split(_delimiters[1]);
				//
				if ( messageTypeAndTriggerEvent != null && messageTypeAndTriggerEvent.Length == 2 )
				{
					_messageType = HL7Utility.ConvertString( messageTypeAndTriggerEvent[0] );
					//
					_triggerEvent = HL7Utility.ConvertString( messageTypeAndTriggerEvent[1] );
				}
			}
			catch(Exception ex)
			{
				_ackErrorCode = AckErrorCodes.SegmentSequenceError;
				var innerException = HL7Utility.GetInnerExceptionFromException(ex);
                // EventLogAppender: note that I changed HL7AtdMessage to HL7BtsMessage, comparing to commented code.
                _eventsLogger.Error("Error processing HL7BtsMessage.\n\n" + innerException);
			}
		}

		#endregion 

		#region Methods

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8646"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8647"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the full HL7 message.
		/// </summary>
		/// <returns>string representation of HL7 message input.</returns>
		public override string GetMessage()
		{
			return Message;
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8644"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8645"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the Message Control ID associated with this message
		/// </summary>
		/// <returns>Message Control ID in MSH segment</returns>
		public override string GetMessageControlID()
		{
			return MessageControlID;
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8642"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8643"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Overriden method to get the Message Type.
		/// </summary>
		/// <returns>Message Type, with Trigger Event, in MSH segment.</returns>
		public override string GetMessageType()
		{
			return MessageType;
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8897"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8898"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// 
		/// </summary>
		public bool IsMessageTypeValid()
		{
			switch( _messageType )
			{
				case BTS:
					return true;
				default:
					return false;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8640"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8641"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method always returns true for this type of message.
		/// </summary>
		/// <returns>boolean</returns>
		public override bool IsResponseRequired()
		{
			return RESPONSE_REQUIRED_INDICATOR;
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/12/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8910"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8911"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// 
		/// </summary>
		public bool IsTriggerEventValid()
		{
			switch( _triggerEvent )
			{
				case O31:
					return true;
				default:
					return false;
			}
		}

		/// <summary>
		/// WriteHL7MessageBody
		/// </summary>
		/// <param name="writer"></param>
		protected override void WriteHL7MessageBody( StreamWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.Write( this.Message );
		}


		/// <summary>
		/// 
		/// </summary>
		private void LoadDataFromHL7Message()
		{
			_delimiters = HL7Utility.ParseGetMessageDelimiters(this.Message);
			//
			string[] hl7Segments = HL7Utility.ParseGetAllMessageSegments( this.Message );
			//
			for (int i = 0 ; i < hl7Segments.Length - 1 ; i++ )
			{
				string[] seg = hl7Segments[i].Split(_delimiters[0]);
				//
				switch( seg[0] )
				{
					case "MSH":
					{
						string [] mshTemp = hl7Segments[i].Split(_delimiters[0]);	
						_msh = new string[mshTemp.Length + 1];
						_msh[0] = mshTemp[0];
						_msh[1] = _delimiters[0].ToString();
						for (int j = 2; j < mshTemp.Length; j++)
						{
							_msh[j] =  mshTemp[j - 1];
						}
						break;
					}
					case "PID":
						_pid = hl7Segments[i].Split(_delimiters[0]);						
						break;

					case "PV1":
						_pv1 = hl7Segments[i].Split(_delimiters[0]);
						break;

					case "ORC":
						_orc = hl7Segments[i].Split(_delimiters[0]);
						break;

					case "BPO":
						_bpo = hl7Segments[i].Split(_delimiters[0]);
						break;

					case "BTX":
						_btx = hl7Segments[i].Split(_delimiters[0]);
						break;

					case "NTE":
						if ( _nte == null )
						{
							_nte = new ArrayList();
						}
						//
						_nte.Add( hl7Segments[i].Split(_delimiters[0]) );
						break;

					default:
						break;
				}
			}
		}

		#endregion

		#region Properties

		///<Developers>
		///	<Developer>Hines OIFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/30/2011</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9200"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9201"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// AckErrorCode
		/// </summary>
		public AckErrorCodes AckErrorCode
		{
			get
			{
				return _ackErrorCode;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8662"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>char array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8663"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Delimiters
		/// </summary>
		public char [] Delimiters
		{
			get
			{
				return _delimiters;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8670"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8671"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MessageType
		/// </summary>
		public string MessageType
		{
			get
			{
				return _messageType;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/12/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8912"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8913"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MessageType
		/// </summary>
		public string TriggerEvent
		{
			get
			{
				return _triggerEvent;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8668"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8669"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Full HL7 message
		/// </summary>
		public string Message
		{
			get
			{
				return _message;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8664"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8665"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Message Control ID
		/// </summary>
		public string MessageControlID
		{
			get
			{
				return _messageControlID;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8660"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8661"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MSH segment
		/// </summary>
		public string[] MSH
		{
			get
			{
				return _msh;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8658"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8659"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PID segment
		/// </summary>
		public string[] PID
		{
			get
			{
				return _pid;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8656"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8657"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PV1 segment
		/// </summary>
		public string[] PV1
		{
			get
			{
				return _pv1;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8654"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8655"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ORC segment
		/// </summary>
		public string[] ORC
		{
			get
			{
				return _orc;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8652"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8653"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BPO segment
		/// </summary>
		public string[] BPO
		{
			get
			{
				return _bpo;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8650"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8651"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BTX segment
		/// </summary>
		public string[] BTX
		{
			get
			{
				return _btx;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8648"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Arraylist</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8649"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// NTE
		/// </summary>
		public ArrayList NTE
		{
			get
			{
				return _nte;
			}
		}


		#endregion
	}
}
